package gov.va.med.mhv.phr.appointments.decorator;

import gov.va.med.mhv.core.util.Precondition;
import gov.va.med.mhv.phr.decorator.DecoratorUtils;
import gov.va.med.mhv.phr.transfer.Appointment;
import gov.va.med.mhv.phr.transfer.ImpreciseDateComposite;

import java.util.Date;

import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang.StringUtils;
import org.displaytag.decorator.TableDecorator;

/*
 * This class is the decorator for the Appointment detail display.
 *
 * @author - Jeff Robinson (Aquilent, Inc.)
 */
public class AppointmentTableDecorator extends TableDecorator{


    protected static final String COMPENSATION_APPT = "This is a Compensation & Pension Appointment";
    protected static final String CP = "COMPENSATION & PENSION";
    protected static final String CANCELLED = "CANCELLED";

    protected static final String BLANK = StringEscapeUtils.escapeHtml("");

    protected static String escapedValueOrBlank(String value) {
        return StringUtils.isBlank(value) ? BLANK : escapedValue(value);
    }

    protected static String escapedValue(String value) {
        return StringEscapeUtils.escapeHtml(value);
    }

    /*
     * This method retrieves and formats for display the Appointment date.
     *
     * @return a String containing the Appointment date/time.
     */
    // Decorator for the Appointment date field. This method converts the imprecise appointment
    // date field into a string that can be presented on the appointment detai table.
    public String getAppointmentDate() {
        ImpreciseDateComposite apptDate = getAppointment().getAppointmentDateComposite();
        if (apptDate == null) {
            return "";
        } else {
        	return DecoratorUtils.formatDateTime(apptDate);
        }
    }

    public Date getAppointmentDateObj() {
    	return AppointmentDecoratorUtil.getAppointmentDate(getAppointment());
    }

    /*
     * This method returns the name of the location of the Appointment
     *
     * @return a String contaning the location.
     */
    public String getLocation() {
        return escapedValue(getAppointment().getLocation());
    }

    /*
     * This method retrieve the name of the Clinic for the Appointment
     *
     * @return a String containing the name of the Clinic
     */
    public String getClinicName() {
        return escapedValue(getAppointment().getClinicName());
    }

    /*
     * This method retrieve the phone number of the Clinic for the Appointment
     *
     * @return a String containing the phone number of the clinic
     */
    public String getClinicPhone() {
        return escapedValue(getAppointment().getClinicPhone());
    }

    /*
     * This method converts the VistA appointment status value into the
     * corresponding MHV appointment status for display in the apppointment
     * details.
     *
     * @return A String containing the MHV status for the appointment.
     */
    public String getAppointmentStatus()
    {
		String appointmentStatus = AppointmentDecoratorUtil.getAppointmentStatus(getAppointment());

		if(appointmentStatus.equalsIgnoreCase(CANCELLED)){
			// <span style="color:#ff0000">Text you want to be red</span>
			//String span1 = "&lt;span style=&quotcolor:#ff0000&quot &gt;";
			//String span2 = "&lt;/span&gt;";

			String span1 = "<span style=\"color:#ff0000\">";
			String span2 = "</span>";
			appointmentStatus = span1 + appointmentStatus + span2;
			return appointmentStatus;
		}else
    	return appointmentStatus;
    }

    /*
     *  This method examines the VistA appointment type and sets the Compensation
     *  and Pension Appointment value accordingly.
     *
     *  @return A String value containing the value to display in the Compensation and Pension field
     */
    public String getCompensationAndPensionStatus()
    {
    	String compAppt = getAppointment().getAppointmentType();

    	if (compAppt != null)
    	{
    		return (compAppt.equalsIgnoreCase(CP)) ? escapedValue(COMPENSATION_APPT): "";
    	}
    	else
    		return "";
    }

    /*
     *  This method retrieve the Imprecise Lab Date and formats it
     *  into a readable date/time string for display.
     *
     *  @return A String value containing the formatted Lab Date and Time
     */
    public String getLabDate() {
        ImpreciseDateComposite labDate = getAppointment().getLabDateComposite();
        if (labDate == null) {
            return "";
        } else {
        	return DecoratorUtils.formatDateTime(labDate);
        }
    }

    /*
     *  This method retrieve the Imprecise EKG Date and formats it
     *  into a readable date/time string for display.
     *
     *  @return A String value containing the formatted EKG Date and Time
     */
    public String getEkgDate() {
        ImpreciseDateComposite ekgDate = getAppointment().getEkgDateComposite();
        if (ekgDate == null) {
            return "";
        } else {
        	return DecoratorUtils.formatDateTime(ekgDate);
        }
    }

    /*
     *  This method retrieve the Imprecise X-Ray Date and formats it
     *  into a readable date/time string for display.
     *
     *  @return A String value containing the formatted X-Ray Data and Time
     */
    public String getXrayDate() {
        ImpreciseDateComposite xrayDate = getAppointment().getXrayDateComposite();
        if (xrayDate == null) {
            return "";
        } else {
        	return DecoratorUtils.formatDateTime(xrayDate);
        }
    }

    /*
     * This method retrieves the Appointment for the selected row in the table.
     *
     * @return   the Appointment object for the current row
     * @see 	 Appointment
     */
    private Appointment getAppointment() {
        Object currentRowObject = getCurrentRowObject();
        Precondition.assertAssignableFrom("currentRowObject", currentRowObject,
            Appointment.class);
        Appointment appt = (Appointment) currentRowObject;
        Precondition.assertNotNull("currentRowObject", appt);
        return appt;
    }

}
